<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

header('Content-Type: application/json');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Function to log debug information
function debug_log($message, $data = null) {
    $log = date('Y-m-d H:i:s') . " - " . $message;
    if ($data !== null) {
        $log .= " - Data: " . print_r($data, true);
    }
    error_log($log);
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    debug_log("User not logged in");
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit();
}

// Get the code from request
$code = isset($_GET['code']) ? trim($_GET['code']) : '';
debug_log("Lookup code received", $code);

if (empty($code)) {
    debug_log("No code provided");
    echo json_encode(['success' => false, 'message' => 'No code provided']);
    exit();
}

try {
    // First, let's check what columns exist in the products table
    $stmt = $pdo->query("DESCRIBE products");
    $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
    debug_log("Available columns in products table", $columns);

    // Look up product by ID (if code is numeric), name, or serial number from product_serial_numbers table
    $query = "
        SELECT p.id, p.name, 
            p.price AS price, 
            p.cost_price,
            p.quantity AS stock_quantity,
            c.require_serials
        FROM products p
        LEFT JOIN categories c ON p.category_id = c.id
        LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id
        WHERE p.id = ? 
        OR p.name LIKE ? 
        OR psn.serial_number = ?
        LIMIT 1
    ";
    
    debug_log("Executing lookup query", [
        'query' => $query,
        'code' => $code
    ]);

    $stmt = $pdo->prepare($query);
    $stmt->execute([$code, "%{$code}%", $code]);
    $product = $stmt->fetch(PDO::FETCH_ASSOC);

    debug_log("Lookup result", $product);

    if ($product) {
        // Ensure all required fields are present
        $product = [
            'id' => $product['id'],
            'name' => $product['name'],
            'price' => isset($product['price']) ? $product['price'] : 0,
            'cost_price' => isset($product['cost_price']) ? $product['cost_price'] : 0,
            'quantity' => isset($product['stock_quantity']) ? $product['stock_quantity'] : 0,
            'require_serials' => isset($product['require_serials']) ? (int)$product['require_serials'] : 0
        ];
        
        debug_log("Processed product", $product);
        
        echo json_encode([
            'success' => true,
            'product' => $product
        ]);
    } else {
        debug_log("Product not found for code", $code);
        echo json_encode([
            'success' => false,
            'message' => 'Product not found'
        ]);
    }
} catch (PDOException $e) {
    debug_log("Database error", $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
} 